#!/usr/bin/env python

import argparse
import os
import subprocess
import sys
import timeit
import platform

def RunCommand(command, print_stdout=False):
  options = {}

  # If using posix make sure started process is in a different
  # process group so that if the started process does something
  # crazy like... killing all processes in its process group 
  # we don't end up being killed ourself.
  if os.name == 'posix':
    options['preexec_fn'] = os.setsid

  processInstance=subprocess.Popen(command,
    stdout=subprocess.PIPE, 
    stderr=subprocess.STDOUT,
    **options
  )
  stdout=processInstance.communicate() #Allow program to run and wait for it to exit.    
  if print_stdout:
    print stdout[0]
  return processInstance.returncode

def PrintResults(tester, kernel, power, runs, exitcode):
  row = [ tester, kernel, '2**%d' % power ]
  row.extend([ str(x) for x in runs ])
  if exitcode == 0:
    avg = sum(runs)/len(runs)
    row.append(str(avg))
  elif exitcode == 124:
    row.append('Timeout')
  else:
    row.append('Fail')
  print ', '.join(row)

def RunExperiment(tester, kernel, args):
  testerCommand = [ 'timeout', str(args.timeout), tester ]
  flags = [ '--platform=%d' % args.platform, '--device=%d' % args.device ]
  if tester == './atester':
    flags.extend(args.atest_flags.split(' '))
  elif tester == './vtester':
    flags.extend(args.vtest_flags.split(' '))
  else:
    assert False

  exitcode = 0
  for power in range(args.minpower,args.maxpower+1):
    if exitcode != 0: break
    nelements = 2**power
   
    runs = []
    for runNumber in range(args.numruns):

      start = timeit.default_timer()
      exitcode |= RunCommand(testerCommand + [ str(nelements), kernel ] + flags)
      if exitcode == 0:
        end = timeit.default_timer()
        runs.append(end-start)
      else: break

    PrintResults(tester, kernel, power, runs, exitcode)

def main():
  parser = argparse.ArgumentParser(description='Run experiments')
  parser.add_argument('--numruns', metavar='N', type=int, help='number of runs per test', default=3)
  parser.add_argument('--minpower', metavar='i', type=int, help='start runs at 2**i elements', default=1)
  parser.add_argument('--maxpower', metavar='j', type=int, help='start runs at 2**j elements', default=15)
  parser.add_argument('--timeout', metavar='t', type=int, help='timeout runs at t seconds', default=3600)
  parser.add_argument('--platform', metavar='p', type=int, help='platform to target', default=0)
  parser.add_argument('--device', metavar='d', type=int, help='device to target', default=0)
  parser.add_argument('--atest-flags', metavar='str', type=str, help='extra flags to pass to atester', default='')
  parser.add_argument('--vtest-flags', metavar='str', type=str, help='extra flags to pass to vtester', default='')
  parser.add_argument('--skip-atest', help='skip atester runs', default=False, action='store_true')
  parser.add_argument('--skip-vtest', help='skip vtester runs', default=False, action='store_true')
  args = parser.parse_args()

  print "# Running experiments"
  print "# ", platform.node()
  print "# ", args

  testers = []
  if not args.skip_atest: testers.append('./atester')
  if not args.skip_vtest: testers.append('./vtester')

  RunCommand(['./atester', '4', 'blelloch.cl', '--clinfo'], print_stdout=True)
  for tester in testers:
    for kernel in ["brentkung.cl", "blelloch.cl", "koggestone.cl", "sklansky.cl"]:
      RunExperiment(tester, kernel, args)

  return 0

if __name__ == '__main__':
  sys.exit(main())
