// #ifdef __cplusplus
// extern "C" {
// #endif

//========================================================================================================================================================================================================200
//	DEFINE/INCLUDE
//========================================================================================================================================================================================================200

//======================================================================================================================================================150
//	COMMON
//======================================================================================================================================================150

#include "common.h"									// (in directory)

//======================================================================================================================================================150
//	UTILITIES
//======================================================================================================================================================150

#include "timer.h"						// (in directory)

//======================================================================================================================================================150
//	KERNEL
//======================================================================================================================================================150

#include "./solver.c"									// (in directory)
#include "opencl.h"						// (in directory)

//======================================================================================================================================================150
//	LIBRARIES
//======================================================================================================================================================150

#include <stdio.h>										// (in path known to compiler)	needed by printf
#include <string.h>										// (in path known to compiler)	needed by strlen
#include <CL/cl.h>										// (in path provided to compiler)	needed by OpenCL types and functions

//======================================================================================================================================================150
//	HEADER
//======================================================================================================================================================150

#include "kernel_gpu_opencl_wrapper.h"					// (in directory)

#include "emi.h"

//======================================================================================================================================================150
//	END
//======================================================================================================================================================150

//========================================================================================================================================================================================================200
//	MAIN FUNCTION
//========================================================================================================================================================================================================200

int 
kernel_gpu_opencl_wrapper(	int xmax,
							int workload,

							fp ***y,
							fp **x,
							fp **params,
							fp *com)
{

	//======================================================================================================================================================150
	//	VARIABLES
	//======================================================================================================================================================150

	long long time0;
	long long time1;
	long long time2;
	long long time3;
	long long time4;
	long long time5;
	long long timecopyin = 0;
	long long timekernel = 0;
	long long timecopyout = 0;
	long long timeother;

	time0 = get_time();

	int i;

	//======================================================================================================================================================150
	//	GPU SETUP
	//======================================================================================================================================================150

	//====================================================================================================100
	//	COMMON VARIABLES
	//====================================================================================================100

	// common variables
	cl_int error;

	//====================================================================================================100
	//	GET PLATFORMS (Intel, AMD, NVIDIA, based on provided library), SELECT ONE
	//====================================================================================================100
  // IMPERIAL EDIT
  cl_platform_id platform;
	cl_device_id device;
	cl_context context;
	cl_command_queue command_queue;
  initCL(platform, device, context, command_queue);
  // END IMPERIAL EDIT

	//====================================================================================================100
	//	CRATE PROGRAM, COMPILE IT
	//====================================================================================================100
  // IMPERIAL EDIT
  cl_program program;
  cl_kernel kernel;
  compileProgram("src/kernel_gpu_opencl.cl", "kernel_gpu_opencl", device, context, program, kernel);
  // END IMPERIAL EDIT

	//====================================================================================================100
	//	INITIAL DRIVER OVERHEAD
	//====================================================================================================100

	// cudaThreadSynchronize();

	time1 = get_time();

	//======================================================================================================================================================150
	//	ALLOCATE MEMORY
	//======================================================================================================================================================150

	//====================================================================================================100
	//	d_initvalu_mem
	//====================================================================================================100

	int d_initvalu_mem;
	d_initvalu_mem = EQUATIONS * sizeof(fp);
	cl_mem d_initvalu;
	d_initvalu = clCreateBuffer(context,					// context
								CL_MEM_READ_WRITE,			// flags
								d_initvalu_mem,				// size of buffer
								NULL,						// host pointer (optional)
								&error );					// returned error
	if (error != CL_SUCCESS) 
		fatal_CL(error, __LINE__);

	//====================================================================================================100
	//	d_finavalu_mem
	//====================================================================================================100

	int d_finavalu_mem;
	d_finavalu_mem = EQUATIONS * sizeof(fp);
	cl_mem d_finavalu;
	d_finavalu = clCreateBuffer(context, 
								CL_MEM_READ_WRITE, 
								d_finavalu_mem, 
								NULL, 
								&error );
	if (error != CL_SUCCESS) 
		fatal_CL(error, __LINE__);

	//====================================================================================================100
	//	d_params_mem
	//====================================================================================================100

	int d_params_mem;
	d_params_mem = PARAMETERS * sizeof(fp);
	cl_mem d_params;
	d_params = clCreateBuffer(	context, 
								CL_MEM_READ_WRITE, 
								d_params_mem, 
								NULL, 
								&error );
	if (error != CL_SUCCESS) 
		fatal_CL(error, __LINE__);

	//====================================================================================================100
	//	d_com_mem
	//====================================================================================================100

	int d_com_mem;
	d_com_mem = 3 * sizeof(fp);
	cl_mem d_com;
	d_com = clCreateBuffer(	context, 
							CL_MEM_READ_WRITE, 
							d_com_mem, 
							NULL, 
							&error );
	if (error != CL_SUCCESS) 
		fatal_CL(error, __LINE__);

  // IMPERIAL EDIT
  cl_mem d_emi_data;
  initEMI(context, command_queue, d_emi_data);
  // END IMPERIAL EDIT

	time2 = get_time();

	//======================================================================================================================================================150
	//	EXECUTION
	//======================================================================================================================================================150

	int status;

	for(i=0; i<workload; i++){

		status = solver(	y[i],
							x[i],
							xmax,
							params[i],
							com,

							d_initvalu,
							d_finavalu,
							d_params,
							d_com,
              d_emi_data,

							command_queue,
							kernel,

							&timecopyin,
							&timekernel,
							&timecopyout);

		if(status !=0){
			printf("STATUS: %d\n", status);
		}

	}

	// // // print results
	// // int k;
	// // for(i=0; i<workload; i++){
		// // printf("WORKLOAD %d:\n", i);
		// // for(j=0; j<(xmax+1); j++){
			// // printf("\tTIME %d:\n", j);
			// // for(k=0; k<EQUATIONS; k++){
				// // printf("\t\ty[%d][%d][%d]=%13.10f\n", i, j, k, y[i][j][k]);
			// // }
		// // }
	// // }

	time3 = get_time();

	//======================================================================================================================================================150
	//	FREE GPU MEMORY
	//======================================================================================================================================================150

	// Release kernels...
	clReleaseKernel(kernel);

	// Now the program...
	clReleaseProgram(program);

	// Clean up the device memory...
	clReleaseMemObject(d_initvalu);
	clReleaseMemObject(d_finavalu);
	clReleaseMemObject(d_params);
	clReleaseMemObject(d_com);

	// Flush the queue
	error = clFlush(command_queue);
	if (error != CL_SUCCESS) 
		fatal_CL(error, __LINE__);

	// ...and finally, the queue and context.
	clReleaseCommandQueue(command_queue);

	// ???
	clReleaseContext(context);

	time4= get_time();

	//======================================================================================================================================================150
	//	DISPLAY TIMING
	//======================================================================================================================================================150

  #ifdef _MSC_VER
  printf("Timing not available on Windows\n");
  #else

	printf("Time spent in different stages of the application:\n");
	printf("%15.12f s, %15.12f % : CPU: GPU SETUP\n", 								(float) (time1-time0) / 1000000, (float) (time1-time0) / (float) (time4-time0) * 100);
	printf("%15.12f s, %15.12f % : CPU: ALLOCATE GPU MEMORY\n", 					(float) (time2-time1) / 1000000, (float) (time2-time1) / (float) (time4-time0) * 100);
	printf("%15.12f s, %15.12f % : GPU: COMPUTATION\n", 							(float) (time3-time2) / 1000000, (float) (time3-time2) / (float) (time4-time0) * 100);

	printf("\tGPU: COMPUTATION Components:\n");
	printf("\t%15.12f s, %15.12f % : GPU: COPY DATA IN\n", 							(float) (timecopyin) / 1000000, (float) (timecopyin) / (float) (time4-time0) * 100);
	printf("\t%15.12f s, %15.12f % : GPU: KERNEL\n", 								(float) (timekernel) / 1000000, (float) (timekernel) / (float) (time4-time0) * 100);
	printf("\t%15.12f s, %15.12f % : GPU: COPY DATA OUT\n", 						(float) (timecopyout) / 1000000, (float) (timecopyout) / (float) (time4-time0) * 100);
	timeother = time3-time2-timecopyin-timekernel-timecopyout;
	printf("\t%15.12f s, %15.12f % : GPU: OTHER\n", 								(float) (timeother) / 1000000, (float) (timeother) / (float) (time4-time0) * 100);

	printf("%15.12f s, %15.12f % : CPU: FREE GPU MEMORY\n", 						(float) (time4-time3) / 1000000, (float) (time4-time3) / (float) (time4-time0) * 100);
	printf("Total time:\n");
	printf("%.12f s\n", 															(float) (time4-time0) / 1000000);

  #endif

	//======================================================================================================================================================150
	//	RETURN
	//======================================================================================================================================================150

	return 0;

	//======================================================================================================================================================150
	//	END
	//======================================================================================================================================================150

}

//========================================================================================================================================================================================================200
//	END
//========================================================================================================================================================================================================200

// #ifdef __cplusplus
// }
// #endif
