#!/usr/bin/env python

from __future__ import print_function

import argparse
import random
import sys

def inject(lines, ntries):
  nlines = len(lines)
  insert = False
  i = 0
  while (not insert or i < ntries):
    linenum = random.randint(0,nlines-1)
    dir = random.randint(0,1)
    if dir == 0: #backwards, make sure { appears
      for l in lines[0:linenum]:
        if '{' in l: insert = True
    else: #forwards, make sure } appears
      for l in lines[linenum:-1]:
        if '}' in l: insert = True
    i += 1

  if insert == False:
    print("Could not insert bugs after {0} tries".format(ntries))
    return None
  else:
    return linenum

def emi_condition(emi_length):
  lo = random.randint(0,emi_length)
  hi = random.randint(lo,emi_length)
  if lo < hi:
    return "emi_data[{0}] < emi_data[{1}]".format(hi,lo)
  elif lo != hi:
    return "emi_data[{0}] != {1}".format(hi,lo)
  else:
    return "emi_data[{0}] != {1}".format(hi,(hi+1)%emi_length)

def main(argv=None):
  if argv is None:
    argv = sys.argv[1:]
  parser = argparse.ArgumentParser(description="EMI injection")
  parser.add_argument("input", type=argparse.FileType('r'), help="Source file to inject EMI blocks")
  parser.add_argument("--output", type=argparse.FileType('w'), default=sys.stdout, help="Direct output to file")
  parser.add_argument("--nbugs", type=int, default=2, help="Number of injections")
  parser.add_argument("--seed", type=int, default=0, help="Seed for randomisation placement")
  parser.add_argument("--ntries", type=int, default=100, help="Number of attempts to insert")
  parser.add_argument("--emi_length", type=int, default=256, help="Number of attempts to insert")
  args = parser.parse_args(argv)

  random.seed(args.seed)
  lines = args.input.readlines()

  nkernels = 0
  for l in lines:
    if "__kernel" in l: nkernels += 1
  if nkernels == 0:
    print("No kernels found in input")
    return 1

  injections = []
  while (len(injections) < args.nbugs):
    i = inject(lines, args.ntries)
    if i == False:
      print("Could not insert bug after n tries")
      return 1
    else:
      injections.append(i)

  print("// IMPERIAL EDIT", file=args.output)
  print("#include \"CLSmith.h\"", file=args.output)
  print("// END IMPERIAL EDIT", file=args.output)
  emi = 0
  for i, l in enumerate(lines):
    l = l.strip('\n')
    whitespace = " " * (len(l) - len(l.lstrip()))
    if i in injections:
      print("// INJECTION", file=args.output)
      print("if ({0}) {{".format(emi_condition(args.emi_length)), file=args.output)
      print("#include \"emi{0}/subst.h\"".format(emi), file=args.output)
      print("#include \"emi{0}/EMI_BLOCK.h\"".format(emi), file=args.output)
      print("}", file=args.output)
      print("// END INJECTION", file=args.output)
      emi += 1
    print(l, file=args.output)
  return 0

if __name__ == '__main__':
  sys.exit(main())
