#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function

import argparse
import filecmp
import os
import shutil
import subprocess
import sys
import textwrap
import yaml
import threading

from runcommon import *

def main(argv=None):
  if argv is None:
    argv = sys.argv[1:]
  parser = argparse.ArgumentParser(
    description="Script to run all variants of emiblocks", 
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=textwrap.dedent('''\
Arguments passed after "--" are passed directly to all application invocations.
Example:

  $ runall.py -- --platform 0 --device 1

Here are valid options that you can pass:
  --platform N --device M            Select a given platform and device
  --name NAME                        Ensure target device contains string 'NAME'
  --set_device_from_name             Select platform/device based on --name flag
  --print_kernel                     Print kernel before compilation
  --no_while                         Wrap EMI blocks in for loop instead of while''')
  )
  parser.add_argument("--emi_lo", metavar="X", type=int, default=0, help="Start emi block testing at X")
  parser.add_argument("--emi_hi", metavar="X", type=int, default=124, help="Stop emi block testing at X")
  parser.add_argument('--timeout', default=60, type=int, help="Timeout for each run")
  parser.add_argument("--verbose", action="store_true", help="Verbosity")
  parser.add_argument("--paranoid", action="store_true", help="Double-check diff")
  parser.add_argument("--no_skip", action="store_true", help="Turn-off skipping of tests (i.e., run all tests regardless of previous results)")
  parser.add_argument("--print_header", action="store_true", help="Print CSV header")
  parser.add_argument('extra_flags', nargs='*', default=[], help="Extra arguments passed directly to the application")
  args = parser.parse_args(argv)

  if args.emi_lo > args.emi_hi:
    print("Bad emi lo-hi combination")
    return 1

  y = read_yaml('runall.cnf')
  if y == None: # error reading configuration
    return 1
  name = y["name"]
  generate = y["generate"]
  expected = y["expected"]
  output = y["output"]
  cmd = y["cmd"]

  if args.print_header:
    print("# benchmark, opt, sub, emiblock, result")
  for opt in range(2):
    if generate:
      # clear expected output
      try:
        os.remove(expected)
      except OSError: # does not exist
        pass
      # create expected output
      r = run_test(cmd, opt, 0, 999, output, output, args)
      if r == ReturnCode.BAD_PLATFORM:
        print("Unrecoverable error: bad platform/device")
        return 1
      if r != ReturnCode.PASS:
        for sub in range(2):
          for emiblock in range(args.emi_lo, args.emi_hi+1):
            print(",".join([name,str(opt),str(sub),str(emiblock),"SKIP_BAD_GEN"]))
        continue
      shutil.copyfile(output, expected)
    for sub in range(2):
      history = []
      for emiblock in range(args.emi_lo, args.emi_hi+1):
        print(",".join([name,str(opt),str(sub),str(emiblock),""]), end="")
        # skip ahead based on history
        history = history[-10:] # at most ten history items
        if not args.no_skip and len(history) == 10 and len(set(history)) == 1 and history[0] != ReturnCode.PASS:
          print("SKIP_REPEAT")
          continue
        r = run_test(cmd, opt, sub, emiblock, expected, output, args)
        history.append(r)
        if r == ReturnCode.PASS:
          print("OKAY")
        elif r == ReturnCode.BAD_DIFF:
          print("BAD_DIFF")
        elif r == ReturnCode.BAD_EXIT_CODE:
          print("BAD_EXIT_CODE")
        elif r == ReturnCode.BAD_TIMEOUT:
          print("BAD_TIMEOUT")
        elif r == ReturnCode.BAD_NO_OUTPUT:
          print("BAD_NO_OUTPUT")
        elif r == ReturnCode.BAD_PLATFORM:
          assert False
        elif r == ReturnCode.BAD_IDENTIFIER:
          print("BAD_IDENTIFIER")
        else:
          assert False
  return 0

if __name__ == '__main__':
  os.environ["CUDA_CACHE_DISABLE"] = "1"
  sys.exit(main())
