import subprocess
import threading
import yaml
import filecmp
import os

class WorkerThread(threading.Thread):
	def __init__(self, timeout, cmd):
		super(WorkerThread, self).__init__()
		self.timeout = timeout
		self.cmd     = cmd
		self.output  = ["", 0]

	def run(self):
		self.process = subprocess.Popen(self.cmd, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, stdin=subprocess.PIPE)
		output, _ = self.process.communicate()
		self.output = output, self.process.returncode

	def start(self):
		threading.Thread.start(self)
		threading.Thread.join(self, self.timeout)
		if self.isAlive():
			self.process.terminate()
			return ["TIMEOUT", 0]
		return self.output

def read_yaml(fname):
  try:
    y = yaml.load(open(fname))
    return y
  except IOError:
    print("Error: Could not find '{0}'".format(fname))
    return None
  except yaml.YAMLError:
    print("Error: Problem parsing '{0}'".format(fname))
    return None
  if "name" not in y or "generate" not in y or "expected" not in y or "output" not in y or "cmd" not in y:
    print("Error: Bad configuration")
    print("Should define name, generate, expected, output and cmd")
    return None

class ReturnCode(object):
  PASS = 0
  BAD_DIFF = 1
  BAD_EXIT_CODE = 2
  BAD_TIMEOUT = 3
  BAD_NO_OUTPUT = 4
  BAD_PLATFORM = 5
  BAD_IDENTIFIER = 6

# True  -> no difference between expected and output
# False -> difference detected
def files_are_equal(expected, output, paranoid=False):
  filecmp._cache.clear()
  filecmp_isequal = filecmp.cmp(output, expected, shallow=False)
  if paranoid and os.name == 'posix':
    try:
      proc = subprocess.Popen(['diff', '-q', output, expected], stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      diff_exitcode = proc.wait()
      diff_isequal = diff_exitcode == 0
      if diff_isequal != filecmp_isequal:
        print("Suspicious diff!")
    except: pass
  return filecmp_isequal

def run_test(cmd, opt, sub, block, expected, output, args):
  # try to clear output
  try:
    os.remove(output)
  except OSError: # does not exist
    pass
  assert not os.path.isfile(output)

  # run the command
  cmdextra = [ '--optimisations', str(opt), '--substitution', str(sub), '--emi_block', str(block) ]
  job = cmd + cmdextra + args.extra_flags
  if args.verbose:
    print("\n{0}".format(' '.join(job)))
  w = WorkerThread(args.timeout, job)
  (stdout, returncode) = w.start()

  if stdout != "TIMEOUT" and args.verbose:
    print(stdout)

  # return status
  if stdout == "TIMEOUT":
    return ReturnCode.BAD_TIMEOUT
  if "error: use of undeclared identifier" in stdout:
    return ReturnCode.BAD_IDENTIFIER
  if "No matching platform/device found" in stdout or "not found in device name" in stdout:
    return ReturnCode.BAD_PLATFORM
  if returncode != 0:
    return ReturnCode.BAD_EXIT_CODE
  if not os.path.isfile(output):
    return ReturnCode.BAD_NO_OUTPUT
  if files_are_equal(expected,output,args.paranoid):
    return ReturnCode.PASS
  else:
    return ReturnCode.BAD_DIFF
