#!/usr/bin/env python
# -*- coding: utf-8 -*-

from __future__ import print_function

import argparse
import filecmp
import os
import shutil
import subprocess
import sys
import textwrap
import yaml

from runcommon import *

def main(argv=None):
  if argv is None:
    argv = sys.argv[1:]
  parser = argparse.ArgumentParser(
    description="Script to run an emiblock variant",
    formatter_class=argparse.RawDescriptionHelpFormatter,
    epilog=textwrap.dedent('''\
Arguments passed after "--" are passed directly to the application invocation.
Example:

  $ runsingle.py -- --platform 0 --device 1

Here are valid options that you can pass:
  --platform N --device M            Select a given platform and device
  --print_kernel                     Print kernel before compilation
  --no_while                         Wrap EMI blocks in for loop instead of while''')
  )
  parser.add_argument("--optimisations", metavar="X", type=int, required=True, help="Optimisations on or off")
  parser.add_argument("--substitutions", metavar="X", type=int, required=True, help="Substitutions on or off")
  parser.add_argument("--emi_block", metavar="X", type=int, default=999, required=True, help="EMI block to run")
  parser.add_argument("--verbose", action="store_true", help="Verbosity")
  parser.add_argument('--timeout', default=60, type=int, help="Timeout for each run")
  parser.add_argument("--paranoid", action="store_true", help="Double-check diff")
  parser.add_argument('extra_flags', nargs='*', default=[], help="Extra arguments passed directly to the application")
  args = parser.parse_args(argv)

  y = read_yaml('runall.cnf')
  if y == None: # error reading configuration
    return 1
  name = y["name"]
  generate = y["generate"]
  expected = y["expected"]
  output = y["output"]
  cmd = y["cmd"]

  if generate:
    # clear expected output
    try:
      os.remove(expected)
    except OSError: # does not exist
      pass
    r = run_test(cmd, args.optimisations, 0, 999, output, output, args)
    if r == ReturnCode.BAD_PLATFORM:
      print("Unrecoverable error: bad platform/device")
      return 1
    if r != ReturnCode.PASS:
      print("# Problem generating expected result (skipping test)")
      return 1
    shutil.copyfile(output, expected)

  r = run_test(cmd, args.optimisations, args.substitutions, args.emi_block, expected, output, args)
  if r == ReturnCode.PASS:
    print("OKAY")
    return 0

  if r == ReturnCode.BAD_DIFF:
    print("BAD_DIFF")
  elif r == ReturnCode.BAD_EXIT_CODE:
    print("BAD_EXIT_CODE")
  elif r == ReturnCode.BAD_TIMEOUT:
    print("BAD_TIMEOUT")
  elif r == ReturnCode.BAD_NO_OUTPUT:
    print("BAD_NO_OUTPUT")
  elif r == ReturnCode.BAD_PLATFORM:
    assert False
  elif r == ReturnCode.BAD_IDENTIFIER:
    print("BAD_IDENTIFIER")
  else:
    assert False
  return 1

if __name__ == '__main__':
  os.environ["CUDA_CACHE_DISABLE"] = "1"
  sys.exit(main())
